<?php

$debug = false;

abstract class Permissions {
    const Read = 1;
    const Write = 2;
    const Delete = 4;
}

abstract class ParameterDirection {
    const Input = 1;
    const Output = 2;
    const InputOutput = 3;
}

class Parameter {
    public string $name;

    public $value;
    
    public int $direction;

    public function __construct(string $name, $value = null, int $direction = ParameterDirection::Input)
    {
        $this->name = $name;
        $this->value = $value;
        $this->direction = $direction;
    }
}

class Parameters {

    public array $array = array();

    public function __construct(Command $command) {
        $this->command = $command;
    }

    public Command $command;

    public function count() { return count($this->array); } 

    public function itemAt(int $index): Parameter {
        return $this->array[$index];
    }

    public function item(string $name): Parameter {

        foreach($this->array as $key => $value)  {
            if (strtolower($value->name) == strtolower($name)) { return $value; }
        }

        throw new Exception("Invalid Argument Exception");
    }

    public function add(Parameter ...$items): Command {

        foreach($items as $myItem) {
            array_push($this->array, $myItem);
        }

        return $this->command;
    }
}

class Command {

    public function __construct(string $commandText) {
        $this->commandText = $commandText;
        $this->parameters = new Parameters($this);
    }
    
    public string $commandText;

    public Parameters $parameters;

    public function execute(mysqli &$cxn = null, $resultCallback = null) : Array {

        global $debug;
        #$debug=true;
        if ($cxn == null) {

            $cxn = \Connection::open();

            $myReturn = Command::execute($cxn, $resultCallback);

            $cxn->close();

            return $myReturn;
        }
        else {
            $myQuery = $this->toString();

            if ($debug) { echo "query: " . $myQuery . "<br />"; }

            if ($cxn->multi_query($myQuery)) {

                $myReturn = [];
                $myRows;
                $myRow;
                $myIndex = 0;

                while (true) {

                    if ($cxn->error != null) { throw new Exception($cxn->error); }

                    $myResult = $cxn->store_result();

                    if ($myResult) 
                    {
                        $myRows = [];

                        while ($myRow = $myResult->fetch_assoc()){
                            if ($resultCallback) { $resultCallback($myIndex, $myRow); }
                            $myRows[] = $myRow;
                        }

                        $myReturn[] = $myRows;

                        $myIndex++;
                    }

                    if (!($cxn->next_result())) { break; }
                }

                if ($cxn->error != null) { throw new Exception($cxn->error); }

                if ($this->hasOutputParameters) {

                    $myOutput = array_pop($myReturn)[0];

                    foreach($myOutput as $key => $value) {
                        $this->parameters->item($key)->value = $myOutput[$key];
                    }
                }

                return $myReturn;
            }
            else if ($cxn->error != null) { 
                throw new Exception($cxn->error);
            }
            else {
                throw new Exception("Unknown Multi Query Error");
            }
        }
    }

    private bool $hasOutputParameters;

    public function toString() : string {

        global $debug;

        $myDeclaration = "";
        $myCall = "";
        $myResult = "";

        $this->hasOutputParameters = false;

        for($myIndex = 0; $myIndex < $this->parameters->count(); $myIndex++) {
        
            $myParameter = $this->parameters->itemAt($myIndex);

            if ($debug) { echo $myParameter->name . ": " . Command::encode($myParameter->value) . "<br />"; }

            if ($myCall != "") { $myCall .= ", "; }

            if (($myParameter->direction & ParameterDirection::Output) == ParameterDirection::Output) {

                $this->hasOutputParameters = true;

                if ($myDeclaration == "")
                { 
                    $myDeclaration = "set ";
                    $myResult = "select ";
                }
                else 
                { 
                    $myDeclaration .= ", "; 
                    $myResult .= ", ";
                }

                $myDeclaration .= "@" . $myParameter->name . "=" . json_encode($myParameter->value, JSON_UNESCAPED_UNICODE);
                $myCall .= "@" . $myParameter->name;
                $myResult .= "@" . $myParameter->name . " as " . $myParameter->name;
            }
            else {
                $myCall .= json_encode($myParameter->value, JSON_UNESCAPED_UNICODE);
            }
        }

        if ($myDeclaration != "") { $myDeclaration .= ";"; }

        return "$myDeclaration call $this->commandText($myCall); $myResult";
    }

    static function fromModel(string $procedureName, object $model) {

        global $debug;

        if ($debug) { echo json_encode($model) . "<br />"; }

        $myReflectionClass = new ReflectionClass(get_class($model));
        $myCommand = new Command($procedureName);
        $myAttributes;
        $myName;

        foreach($myReflectionClass->getProperties() as $myProperty) {
            
            $myName = $myProperty->getName();

            $myAttributes = $myProperty->getAttributes(ParameterAttribute::class);
            $myArguments;
            $myName;

            foreach($myAttributes as $myAttribute) {
                
                if ($myAttribute->getName() == "ParameterAttribute") {
                    $myArguments = $myAttribute->getArguments();

                    if ($debug) { echo "property: $myName, " . $myProperty->getValue($model) . "<br />"; }
                    
                    switch($myProperty->getType()) {

                        default:
                            $myCommand->parameters->add(new Parameter($myName, $myProperty->getValue($model), array_getValue($myArguments, "direction", ParameterDirection::Input)));
                            break;
                    }
                }
            }
        }

        return $myCommand;
    }
}

#[Attribute]
class ParameterAttribute {

    public int $direction = ParameterDirection::Input;
}

class Http {

    static array $ALLOWED_ORIGINS = ["http://roman", "https://roman", "http://localhost", "https://localhost", "http://127.0.0.1", "https://127.0.0.1"];
    static string $ALLOWED_HEADERS = "Accept,Content-Type,Authorization";
    static bool $ALLOW_CREDENTIALS = true;

    static function isOptionsRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "OPTIONS"; }

    static function isGetRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "GET"; }

    static function isPostRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "POST"; }

    static function isDeleteRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "DELETE"; }

    static function isPutRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "OPTIONS"; }

    static function isActionRequest() : bool { return $_SERVER["REQUEST_METHOD"] != "OPTIONS"; }

    static function isUriRequest() : bool { return $_SERVER["REQUEST_METHOD"] == "GET"; }

    static function isFormRequest() : bool { return $_SERVER["REQUEST_METHOD"] != "OPTIONS" && $_SERVER["REQUEST_METHOD"] != "GET" && str_contains(strtolower($_SERVER["CONTENT_TYPE"]), "application/x-www-form-urlencoded"); }

    static function isBodyRequest() : bool { return $_SERVER["REQUEST_METHOD"] != "OPTIONS" && $_SERVER["REQUEST_METHOD"] != "GET"; }

    static function responseCorsHeaders(array $origins, string $headers = "*", string $methods = "*", bool $credentials = null): void {

        $myRequestOrigin = array_getValue($_SERVER, "HTTP_ORIGIN");

        if ($myRequestOrigin) {
            foreach ($origins as $myOrigin) {
                if ($myOrigin == "*" || str_starts_with($myRequestOrigin, $myOrigin)) {
                    header("Access-Control-Allow-Origin: " . $myRequestOrigin);
                    break;
                }
            }
        }
        header("Access-Control-Allow-Headers: " . $headers);
        header("Access-Control-Allow-Methods: " . $methods);
        if ($credentials != null) { header("Access-Control-Allow-Credentials: " . json_encode($credentials)); }
    }

    static function responseDefaultHeaders(string $methods = "*") : void { Http::responseCorsHeaders(Http::$ALLOWED_ORIGINS, Http::$ALLOWED_HEADERS, $methods, Http::$ALLOW_CREDENTIALS); }

    static function fromUri(string $className) {

        global $debug;

        $myReturn = new $className();

        $myReflectionClass = new ReflectionClass($className);

        foreach($_GET as $myName => $myParameter) {

            if ($myReflectionClass->hasProperty($myName)) {

                $myProperty = $myReflectionClass->getProperty($myName);
                $myType = $myProperty->getType()->getName();
                $myValue = array_getValue($_GET, $myName, null);

                if ($myType === "bool") { 
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else if ($myValue === "1" || strtolower($myValue) === "on" || strtolower($myValue) === "true") { $myProperty->setValue($myReturn, true); }
                    else { $myProperty->setValue($myReturn, false); }
                }
                else if ($myType === "int" || $myType == "float") {
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else { $myProperty->setValue($myReturn, (float)$myValue); }
                }
                else if ($myType === "DateTime") {
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else { $myProperty->setValue($myReturn, date_create($myValue)); }
                }
                else { 
                    $myProperty->setValue($myReturn, $myValue); 
                }

                if ($debug) { echo "$myName: $myType, $myValue<br />"; }
            }
        }

        if ($debug) { var_dump($myReturn); echo "<br />"; }

        return $myReturn;
    }

    static function fromBody(string $className) {

        global $debug;

        $myReturn = new $className();

        $myReflectionClass = new ReflectionClass($className);

        foreach($_POST as $myName => $myParameter) {

            if ($myReflectionClass->hasProperty($myName)) {

                $myProperty = $myReflectionClass->getProperty($myName);
                $myType = $myProperty->getType()->getName();
                $myValue = array_getValue($_POST, $myName, null);

                if ($myType === "bool") { 
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else if ($myValue === "1" || strtolower($myValue) === "on" || strtolower($myValue) === "true") { $myProperty->setValue($myReturn, true); }
                    else { $myProperty->setValue($myReturn, false); }
                }
                else if ($myType === "int" || $myType == "float") {
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else { $myProperty->setValue($myReturn, (float)$myValue); }
                }
                else if ($myType === "DateTime") {
                    if ($myValue == "") { $myProperty->setValue($myReturn, null); }
                    else { $myProperty->setValue($myReturn, date_create($myValue)); }
                }
                else { 
                    $myProperty->setValue($myReturn, $myValue); 
                }

                if ($debug) { echo "$myName: $myType, $myValue<br />"; }
            }
        }

        if ($debug) { var_dump($myReturn); echo "<br />"; }

        return $myReturn;
    }

    static function fromJson(string $className, string $json = null) {
        
        $myReturn = new $className();

        $json = $json ?? file_get_contents("php://input");
        $mySource = json_decode($json); //, false, 512, JSON_UNESCAPED_UNICODE);

        //var_dump($mySource);

        //echo json_encode($mySource, JSON_UNESCAPED_UNICODE);

        if ($mySource) {

            $myReflectionClass = new ReflectionClass($className);
            $myProperty;

            foreach($mySource as $myKey => $myValue) {

                if ($myReflectionClass->hasProperty($myKey)) {

                    $myProperty = $myReflectionClass->getProperty($myKey);

                    switch ($myProperty->getType()) {

                        case "DateTime":
                        case "?DateTime":
                            $myReturn->$myKey = date_toUtcDateTime($myValue);
                            break;

                        default:
                            $myReturn->$myKey = $myValue;
                            break;
                    }
                }
            }
        }

        return $myReturn;
    }

    static function responseOk(mixed $data, int $code = 200) {
        http_response_code($code);
        if (is_string($data)) { echo $data; }
        else { echo json_encode($data); }
    }

    static function responseBadRequest(mixed $ex) {
        Http::responseError($ex, 400);
    }

    static function responseError(mixed $ex, int $code = 400) {
        http_response_code($code);
        if (is_string($ex)) { echo $ex; }
        else { echo $ex->getMessage(); }
    }
}

function array_getValue(array $array, string $key, $default = null) : mixed {
    if (array_key_exists($key, $array)) { return $array[$key]; }
    else { return $default; }
}

function date_toUtcDateTime(null | string | DateTime $utcDateTime): null | DateTime {
    return $utcDateTime ? new DateTime($utcDateTime, new DatetimeZone("UTC")) : null;
}

function date_toLocalDateTime(null | string | DateTime $utcDateTime) : null | DateTime {
    
    if ($utcDateTime) {

        if (gettype($utcDateTime) == "string") { $utcDateTime = date_toUtcDateTime($utcDateTime); }

        return  $utcDateTime->setTImeZone(new DateTimeZone(date_default_timezone_get()));
    }
    else {
        return null;
    }
}

function date_toLocalDateTimeString(null | string | DateTime $utcDateTime, string $format = 'Y-m-d H:i:s') : null | string {
    return $utcDateTime ? date_toLocalDateTime($utcDateTime)->format($format) : null;
}

/*
    function mariaGetProcedureCall(string $procedureName, array $parameters = []) {

        $myReturn = "";

        foreach($parameters as $myParameter) {

            if (strlen($myReturn) > 0) { $myReturn .= ", "; }

            if ($myParameter["value"] === null) { $myReturn .= "null"; }
            else if ($myParameter["value"] === false) { $myReturn .= "0"; }
            else if (array_key_exists("nq", $myParameter) && $myParameter["nq"]) { $myReturn .= strval($myParameter["value"]); }
            else { $myReturn .= "'" . strval($myParameter["value"]) . "'"; }
        }

        return "call " . $procedureName . "(" . $myReturn . ")";
    }

    function mariaGetArray ($result) {

        $myRows = array();

        while ($myRow = mysqli_fetch_assoc($result)) {
            $myRows[] = $myRow;
        }

        #echo json_encode($myRows);

        if (count($myRows) == 1) {

            if (array_key_exists("Level", $myRows[0]) && $myRows[0]["Level"] === "Error" ){
                throw new Exception($myRows[0]["Message"]);
                //Http::responseBadRequest($myRows[0]["Message"]);
                //return null;
            }
            else{
                return $myRows;
            }
        }
        else{
            return $myRows;
        }
    }
*/
?>